//*************************************************************************************************
//
//	Description:
//		basic_instanced.fx
//		All the features of basic.fx, but for instanced rendering
//		When used in 3DSMax, it should behave like normal basic.fx
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		AMurray					13/07/2007  0.1           Created
//		TMann						18/07/2007	0.3						PS3 version
//		TNettleship     07/24/2007  0.4           Made sure samplers aren't using anisotropic filtering.
//		TNettleship     08/17/2007  0.5           Removed texture bias, fresnel term and premultiplied alpha.
//		TNettleship     08/30/2007  0.6           Added alphatest technique, mods for atoc support.
//		TNettleship     10/05/2007  0.7           Fixed envmap code.
//		TNettleship			10/23/2007	0.8						Converted to on-load rendermode behaviour binding.
//	<TABLE>
//
//*************************************************************************************************

#define _SSAO_READY_

#ifndef _3DSMAX_
#define	INSTANCING
#endif

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

#define MAX_ENVMAP_BIAS	6.0f



//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif

#if defined(INSTANCING) && defined(_XBOX)
float4 instance_data		// x=no of indices per instance, y=start of index data within IB
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#endif

#if defined(INSTANCING) && defined(_PS3_)
float4x4 instanceXForm
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;

float4x4 instanceworldviewproj
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#endif


//
// Transforms
//
#ifdef INSTANCING
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else

#if defined( _3DSMAX_ )
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;
#endif


//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
>;

#endif


//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;

texture specularTexture : TEXTURE							// Specular colour in RGB, shininess in alpha
<
	string UIName = "Spec Tex {UV1}";
	bool appEdit = true;
>;

SPECIALISATION_PARAM( useNormalMap, "Use normal map?", "NORMAL_MAPPING" )	// TRUE if the normal map is to be used in lighting
DECLARE_DEPENDENT_VERTEX_STREAM( tangentDependency, tangent, TANGENT, useNormalMap )
DECLARE_DEPENDENT_VERTEX_STREAM( binormalDependency, binormal, BINORMAL, useNormalMap )

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
DEPENDENT_TEXTURE_PARAM( normalTexture, "Normal Tex {UV1}", useNormalMap )
DEPENDENT_BOOL_PARAM( specialNormalMap, "Special normal map format?", useNormalMap )
#endif

SPECIALISATION_PARAM( useEnvironmentMap, "Use environment map?", "ENV_MAPPING" )	// TRUE if the environment map is to be used

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_CUBE_TEXTURE_PARAM( environmentTexture, "Env Texture", useEnvironmentMap, false )
#endif

float minSpecPower
<
	string UIName = "Min Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 1.0f;

float maxSpecPower
<
	string UIName = "Max Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 32.0f;

float globalSpecularFactor
<
	string UIName = "Specular Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalEMapFactor, "EMap Factor", useEnvironmentMap, 0.0f, 1.0f, 1.0f )
#endif

//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D specularMap : SAMPLER 
<
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
samplerCUBE environmentMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="environmentTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	string AddressW  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < environmentTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	AddressW  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	AddressW  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif



//
// Functions
//


//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float3 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float3 alpha		: TEXCOORD2;													// Vertex alpha
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#endif
	float3 normal   : NORMAL;															// Object space normal

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent  : TANGENT;														// Object space tangent
	float3 binormal : BINORMAL;														// Object space normal
#endif

#if !defined(_PS3_)	// Not needed on PS3
#ifdef INSTANCING
	float4 world1 : TEXCOORD8;
	float4 world2 : TEXCOORD9;
	float4 world3 : TEXCOORD10;
	float4 world4 : TEXCOORD11;
#endif
#endif
};


struct SHADOWGEN_VSINPUT
{
	float3 position : POSITION;														// Object space position
	float2 texCoord : TEXCOORD0;													// Texture coords

#if !defined(_PS3_)
#ifdef INSTANCING
	float4 world1 : TEXCOORD8;
	float4 world2 : TEXCOORD9;
	float4 world3 : TEXCOORD10;
	float4 world4 : TEXCOORD11;
#endif
#endif
};

struct SOLID_SHADOWGEN_VSINPUT
{
	float3 position : POSITION;														// Object space position

#if !defined(_PS3_)
#ifdef INSTANCING
	float4 world1 : TEXCOORD8;
	float4 world2 : TEXCOORD9;
	float4 world3 : TEXCOORD10;
	float4 world4 : TEXCOORD11;
#endif
#endif
};



// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;												// Normal vector (world space)
	float4 eye				: TEXCOORD2;												// Eye vector (world space)

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD4;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD5;												// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_VS( 6 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_VS( 4 )
#endif
};

struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD1;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
};

struct SHADOWGEN_VSOUTPUT
{
	float4 position			: POSITION;

#ifdef NEEDS_SHADOW_COORDS
	float4 shadowCoord	: TEXCOORD1;
#endif
};

struct SHADOWGEN_T0_VSOUTPUT	// for translucent shadows that need TexCoord0
{
	float4 position			: POSITION;

	float2 texCoord			: TEXCOORD0;
#ifdef NEEDS_SHADOW_COORDS
	float4 shadowCoord	: TEXCOORD1;
#endif
};


struct ZPRIMEDOF_VSOUTPUT
{
	float4 position			: POSITION;
	float2 texCoord			: TEXCOORD0;
	float4 coords			: TEXCOORD1;
};


//-----------------------------------------------------------------------
//
// Vertex shader code
//

#ifdef _POST_PARSE_
VSOUTPUT BasicVertexShader( VSINPUT _input )
{
#else
#if defined(INSTANCING) && defined(_XBOX)
VSOUTPUT BasicVertexShader( int Index : INDEX )
{
    // Compute the instance index
    int iInstanceIndex = ( Index + 0.5 ) / instance_data.x;

    // Compute the mesh index - this is the index to fetch within the current instance
    int iMeshIndex = Index - ( iInstanceIndex * instance_data.x ) + instance_data.y;

	VSINPUT _input;
	float4 fposition;
	float4 fcolour;
	float4 ftexCoord;
	float4 fnormal;
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	float4 ftangent;
	float4 fbinormal;
#endif
	float4 fworld1;
	float4 fworld2;
	float4 fworld3;
	float4 fworld4;
    asm
    {
        vfetch fposition, iMeshIndex, position0;
        vfetch fcolour, iMeshIndex, color0;
        vfetch ftexCoord, iMeshIndex, texcoord0;
        vfetch fnormal, iMeshIndex, normal0;
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
        vfetch ftangent, iMeshIndex, tangent0;
        vfetch fbinormal, iMeshIndex, binormal0;
#endif

        vfetch fworld1, iInstanceIndex, texcoord8, INSTANCES_CACHE;
        vfetch fworld2, iInstanceIndex, texcoord9, INSTANCES_CACHE;
        vfetch fworld3, iInstanceIndex, texcoord10, INSTANCES_CACHE;
        vfetch fworld4, iInstanceIndex, texcoord11, INSTANCES_CACHE;
    };
	_input.position = fposition.xyz;
	_input.colour = fcolour;
	_input.texCoord = ftexCoord.xy;
	_input.normal = fnormal.xyz;
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	_input.tangent = ftangent.xyz;
	_input.binormal = fbinormal.xyz;
#endif
	_input.world1 = fworld1;
	_input.world2 = fworld2;
	_input.world3 = fworld3;
	_input.world4 = fworld4;
#else
VSOUTPUT BasicVertexShader( VSINPUT _input )
{
#endif
#endif

#ifdef INSTANCING
#ifdef _PS3_
	float4x4	world=instanceXForm;
#else
	float4x4	world = { _input.world1, _input.world2, _input.world3, _input.world4 };
#endif	
#endif

#if !defined( _3DSMAX_ )
#if defined (_PS3_)
	float4x4	worldviewproj = instanceworldviewproj;
#else
	float4x4	worldviewproj = mul( world, viewproj );
#endif	
#endif

	VSOUTPUT _output;

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	_output.normal   = normal;

	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;
		_output.binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	return _output;
}





#ifdef _POST_PARSE_
VSOUTPUT_LD BasicLowDetailVertexShader( VSINPUT _input )
{
#else
#if defined(INSTANCING) && defined(_XBOX)
VSOUTPUT_LD BasicLowDetailVertexShader( int Index : INDEX )
{
    // Compute the instance index
    int iInstanceIndex = ( Index + 0.5 ) / instance_data.x;

    // Compute the mesh index - this is the index to fetch within the current instance
    int iMeshIndex = Index - ( iInstanceIndex * instance_data.x ) + instance_data.y;

	VSINPUT _input;
	float4 fposition;
	float4 fcolour;
	float4 ftexCoord;
	float4 fnormal;
	float4 fworld1;
	float4 fworld2;
	float4 fworld3;
	float4 fworld4;
    asm
    {
        vfetch fposition, iMeshIndex, position0;
        vfetch fcolour, iMeshIndex, color0;
        vfetch ftexCoord, iMeshIndex, texcoord0;
        vfetch fnormal, iMeshIndex, normal0;
        vfetch fworld1, iInstanceIndex, texcoord8, INSTANCES_CACHE;
        vfetch fworld2, iInstanceIndex, texcoord9, INSTANCES_CACHE;
        vfetch fworld3, iInstanceIndex, texcoord10, INSTANCES_CACHE;
        vfetch fworld4, iInstanceIndex, texcoord11, INSTANCES_CACHE;
    };
	_input.position = fposition.xyz;
	_input.colour = fcolour;
	_input.texCoord = ftexCoord.xy;
	_input.normal = fnormal.xyz;
	_input.world1 = fworld1;
	_input.world2 = fworld2;
	_input.world3 = fworld3;
	_input.world4 = fworld4;
#else
VSOUTPUT_LD BasicLowDetailVertexShader( VSINPUT _input )
{
#endif
#endif

#ifdef INSTANCING
#ifdef _PS3_
	float4x4	world=instanceXForm;
#else
	float4x4	world = { _input.world1, _input.world2, _input.world3, _input.world4 };
#endif	
#endif

#if !defined( _3DSMAX_ )
#if defined (_PS3_)
	float4x4	worldviewproj = instanceworldviewproj;
#else
	float4x4	worldviewproj = mul( world, viewproj );
#endif	
#endif

	VSOUTPUT_LD _output;

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// get normal in world space and do lighting
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Do lighting
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	// keep alpha value from the vertex
#if defined( _3DSMAX_ )
	_output.colour.a = _input.alpha.r;
#else
	_output.colour.a = _input.colour.a;
#endif

	return _output;
}



#if defined(INSTANCING) && defined(_XBOX)
SHADOWGEN_T0_VSOUTPUT BasicInstancedTranslucentShadowGenVertexShader( int Index : INDEX )
{
    // Compute the instance index
    int iInstanceIndex = ( Index + 0.5 ) / instance_data.x;

    // Compute the mesh index - this is the index to fetch within the current instance
    int iMeshIndex = Index - ( iInstanceIndex * instance_data.x ) + instance_data.y;

	VSINPUT _input;
	float4 fposition;
	float4 ftexcoord;
	float4 fworld1;
	float4 fworld2;
	float4 fworld3;
	float4 fworld4;
    asm
    {
        vfetch fposition, iMeshIndex, position0;
        vfetch ftexcoord, iMeshIndex, texcoord0;

        vfetch fworld1, iInstanceIndex, texcoord8, INSTANCES_CACHE;
        vfetch fworld2, iInstanceIndex, texcoord9, INSTANCES_CACHE;
        vfetch fworld3, iInstanceIndex, texcoord10, INSTANCES_CACHE;
        vfetch fworld4, iInstanceIndex, texcoord11, INSTANCES_CACHE;
    };
	_input.position = fposition.xyz;
	_input.texCoord = ftexcoord.xy;
	_input.world1 = fworld1;
	_input.world2 = fworld2;
	_input.world3 = fworld3;
	_input.world4 = fworld4;
#else
SHADOWGEN_T0_VSOUTPUT BasicInstancedTranslucentShadowGenVertexShader( SHADOWGEN_VSINPUT _input )
{
#endif
	SHADOWGEN_T0_VSOUTPUT _output;

#ifdef INSTANCING
#ifdef _PS3_
	float4x4	world=instanceXForm;
#else
	float4x4	world = { _input.world1, _input.world2, _input.world3, _input.world4 };
#endif	
#endif

#if !defined( _3DSMAX_ )
#if defined (_PS3_)
	float4x4	worldviewproj = instanceworldviewproj;
#else
	float4x4	worldviewproj = mul( world, viewproj );
#endif
#endif

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Clamp geometry that is behind the camera to the near plane so that it still renders.
	// We use an orthogonal projection, so doing this will not distort the shadow caster.
	CLAMP_SHADOW_Z;

#ifdef NEEDS_SHADOW_COORDS
	OUTPUT_SHADOW_COORDS;
#endif

	_output.texCoord = _input.texCoord;

	return _output;
}


#if defined(INSTANCING) && defined(_XBOX)
SHADOWGEN_VSOUTPUT BasicInstancedSolidShadowGenVertexShader( int Index : INDEX )
{
    // Compute the instance index
    int iInstanceIndex = ( Index + 0.5 ) / instance_data.x;

    // Compute the mesh index - this is the index to fetch within the current instance
    int iMeshIndex = Index - ( iInstanceIndex * instance_data.x ) + instance_data.y;

	VSINPUT _input;
	float4 fposition;
	float4 ftexcoord;
	float4 fworld1;
	float4 fworld2;
	float4 fworld3;
	float4 fworld4;
    asm
    {
        vfetch fposition, iMeshIndex, position0;
        vfetch ftexcoord, iMeshIndex, texcoord0;

        vfetch fworld1, iInstanceIndex, texcoord8, INSTANCES_CACHE;
        vfetch fworld2, iInstanceIndex, texcoord9, INSTANCES_CACHE;
        vfetch fworld3, iInstanceIndex, texcoord10, INSTANCES_CACHE;
        vfetch fworld4, iInstanceIndex, texcoord11, INSTANCES_CACHE;
    };
	_input.position = fposition.xyz;
	_input.world1 = fworld1;
	_input.world2 = fworld2;
	_input.world3 = fworld3;
	_input.world4 = fworld4;
#else
SHADOWGEN_VSOUTPUT BasicInstancedSolidShadowGenVertexShader( SOLID_SHADOWGEN_VSINPUT _input )
{
#endif
	SHADOWGEN_VSOUTPUT _output;

#ifdef INSTANCING
#ifdef _PS3_
	float4x4	world=instanceXForm;
#else
	float4x4	world = { _input.world1, _input.world2, _input.world3, _input.world4 };
#endif	
#endif

#if !defined( _3DSMAX_ )
#if defined (_PS3_)
	float4x4	worldviewproj = instanceworldviewproj;
#else
	float4x4	worldviewproj = mul( world, viewproj );
#endif
#endif

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Clamp geometry that is behind the camera to the near plane so that it still renders.
	// We use an orthogonal projection, so doing this will not distort the shadow caster.
	CLAMP_SHADOW_Z;

#ifdef NEEDS_SHADOW_COORDS
	OUTPUT_SHADOW_COORDS;
#endif

	return _output;
}



#if defined(INSTANCING) && defined(_XBOX)
ZPRIMEDOF_VSOUTPUT BasicInstancedZPrimeDOFVertexShader( int Index : INDEX )
{
    // Compute the instance index
    int iInstanceIndex = ( Index + 0.5 ) / instance_data.x;

    // Compute the mesh index - this is the index to fetch within the current instance
    int iMeshIndex = Index - ( iInstanceIndex * instance_data.x ) + instance_data.y;

	VSINPUT _input;
	float4 fposition;
	float4 ftexcoord;
	float4 fworld1;
	float4 fworld2;
	float4 fworld3;
	float4 fworld4;
    asm
    {
        vfetch fposition, iMeshIndex, position0;
        vfetch ftexcoord, iMeshIndex, texcoord0;

        vfetch fworld1, iInstanceIndex, texcoord8, INSTANCES_CACHE;
        vfetch fworld2, iInstanceIndex, texcoord9, INSTANCES_CACHE;
        vfetch fworld3, iInstanceIndex, texcoord10, INSTANCES_CACHE;
        vfetch fworld4, iInstanceIndex, texcoord11, INSTANCES_CACHE;
    };
	_input.position = fposition.xyz;
	_input.texCoord = ftexcoord.xy;
	_input.world1 = fworld1;
	_input.world2 = fworld2;
	_input.world3 = fworld3;
	_input.world4 = fworld4;
#else
ZPRIMEDOF_VSOUTPUT BasicInstancedZPrimeDOFVertexShader( SHADOWGEN_VSINPUT _input )
{
#endif
	ZPRIMEDOF_VSOUTPUT _output;

#ifdef INSTANCING
#ifdef _PS3_
	float4x4	world=instanceXForm;
#else
	float4x4	world = { _input.world1, _input.world2, _input.world3, _input.world4 };
#endif	
#endif

#if !defined( _3DSMAX_ )
#if defined (_PS3_)
	float4x4	worldviewproj = instanceworldviewproj;
#else
	float4x4	worldviewproj = mul( world, viewproj );
#endif	
#endif

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );
	_output.coords = _output.position;
	_output.texCoord = _input.texCoord;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure
#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

struct PSINPUT
{
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;												// Normal vector (world space)
	float4 eye				: TEXCOORD2;												// Eye vector (world space)
	float3 tangent		: TEXCOORD4;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD5;												// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 6 )
};

#else

struct PSINPUT
{
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1_centroid;								// Normal vector (world space)
	float4 eye				: TEXCOORD2_centroid;								// Eye vector (world space)

#if defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD4_centroid;								// Tangent vector (world space)
	float3 binormal		: TEXCOORD5_centroid;								// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 6 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_PS( 4 )
#endif
	DECLARE_SHADOW_PS_INPUTS
};

#endif


#ifdef NEEDS_SHADOW_COORDS	// if this is not defined, SHADOWGEN_PSINPUT is an empty struct
struct SHADOWGEN_PSINPUT	// this one does not have the TexCoord0
{
	float4 shadowCoord	: TEXCOORD1;
};
#endif	

struct SHADOWGEN_T0_PSINPUT	// this one has TexCoord0 to sample diffuse texture
{
	float2 texCoord			: TEXCOORD0;
#ifdef NEEDS_SHADOW_COORDS
	float4 shadowCoord	: TEXCOORD1;
#endif	
};

// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT BasicFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )
	
	float	globalSpecularFactorValue = globalSpecularFactor;
	float	minSpecPowerValue = minSpecPower;
	float	maxSpecPowerValue = maxSpecPower;

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	float4 specularTexColour = tex2D( specularMap, _input.texCoord );

  // Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// Do tangent space normal mapping if required
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		// Normalise the input tangent and binormal vectors
		float3 tangent = normalize( _input.tangent );
		float3 binormal = normalize( _input.binormal );

		// Fetch and decode the map normal
		float4 normalMapColour = tex2D( normalMap, _input.texCoord );

		float3 normalFromMap;
		if ( specialNormalMap )
		{
			normalFromMap = normalize( ( normalMapColour.agb * 2.0f ) - 1.0f );
		}
		else
		{
			normalFromMap = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );
		}

		// Perturb the tangent space normal by the normal map
		normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
		normal = normalize( normal );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap )
#if defined( _3DSMAX_ ) || !defined( NORMAL_MAPPING )
		// No normal map, so use interpolated normal and constant specular strength
		normal = TSnormal;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// If environment mapping is switched on
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
		// Calculate the reflection vector
		float3 worldReflectionVector = reflect( -eye, normal );

		// Fetch the environment map colour using the world coords vector
#ifdef _3DSMAX_
		float4 environmentTexColour = texCUBElod( environmentMap, float4( worldReflectionVector.xzy, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#else
		float4 environmentTexColour = texCUBElod( environmentMap, float4( worldReflectionVector, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#endif

		// Calculate envmap colour and add to diffuse
		accumulatedColour += specularTexColour * environmentTexColour * globalEMapFactor;

		// Switch specular lighting off
		globalSpecularFactorValue = 0.0f;
#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	accumulatedColour.w = diffuseTexColour.w;
	_output.Colour = CalculateOutputPixel(accumulatedColour);
	
	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT BasicLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

	// Apply lighting
	float4 accumulatedColour = diffuseTexColour * _input.colour;
	accumulatedColour.w = diffuseTexColour.w;
	_output.Colour = CalculateLowDetailOutputPixel( accumulatedColour );

	return _output;
}

PSOUTPUT BasicInstancedTranslucentShadowGenFragmentShader( SHADOWGEN_T0_PSINPUT _input )
{
	PSOUTPUT output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

#if defined(NEEDS_SHADOW_COORDS)
	CALC_SHADOWMAP_DEPTH( output.Colour, _input.shadowCoord );
#else
  output.Colour.rgb = 0.0f;
#endif
	output.Colour.a = diffuseTexColour.a;
	SHADOWMAP_PS_ALPHATEST( output.Colour.a, 0.25f );
	
	return output;	
}



PSOUTPUT BasicInstancedSolidShadowGenFragmentShader
( 
#ifdef NEEDS_SHADOW_COORDS
SHADOWGEN_PSINPUT _input
#endif
 )
{
#if !defined(NEEDS_SHADOW_COORDS)
	PSOUTPUT output;
	
	output.Colour=0.0f;
	
	return output;
#else
	PSOUTPUT output;

	CALC_SHADOWMAP_DEPTH( output.Colour, _input.shadowCoord );

	return output;
#endif	
}



PSOUTPUT BasicInstancedZPrimeDOFFragmentShader( ZPRIMEDOF_VSOUTPUT _input )
{
	PSOUTPUT output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	output.Colour = _input.coords.z / _input.coords.w;
	output.Colour.a = diffuseTexColour.a;

	return output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT BasicInstancedOverdrawFragmentShader( PSINPUT _input )
{
	PSOUTPUT output;

	output.Colour = 0.05f;

	return output;
}




//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Basic
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Basic";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour = "ERMB_RENDER";
	string zprimeDOFTechnique = "_Basic_Instanced_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour = "ERMB_RENDER";
	string shadowGenTechnique = "_Basic_Instanced_Solid_ShadowGen";
	int    shadowGenDeferredID	= 0;
	string overdrawBehaviour = "ERMB_RENDER";
	string overdrawTechnique = "_Basic_Instanced_Overdraw";
	int    overdrawDeferredID	= 0;
	string meshVisBehaviour   = "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Basic_LowDetail";
	int    lowDetailDeferredID = 0;
	bool   instancedShader = true;
	bool   disableAmbientShadow = true;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx BasicVertexShader();
		PixelShader = compile sce_fp_rsx BasicFragmentShader();
#else
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
#endif
	}
}


technique Basic_AlphaTest
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Basic_AlphaTest";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour = "ERMB_RENDER";
	string zprimeDOFTechnique = "_Basic_Instanced_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour = "ERMB_RENDER";
	string shadowGenTechnique = "_Basic_Instanced_ShadowGen";
	int    shadowGenDeferredID	= 0;
	string overdrawBehaviour = "ERMB_RENDER";
	string overdrawTechnique = "_Basic_Instanced_Overdraw";
	int    overdrawDeferredID	= 0;
	string meshVisBehaviour   = "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Basic_LowDetail";
	int    lowDetailDeferredID = 0;
	bool   instancedShader = true;
	bool   disableAmbientShadow = true;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
		bool AlphaTestEnable = true;
		int	 AlphaRef = 64;
		string AlphaFunc = "GreaterEqual";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
		AlphaTestEnable = true;
		AlphaRef = 0x40;
		AlphaFunc = GreaterEqual;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx BasicVertexShader();
		PixelShader = compile sce_fp_rsx BasicFragmentShader();
#else
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
#endif
	}
}


technique Basic_LowDetail
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Basic_LowDetail";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour = "ERMB_RENDER";
	string zprimeDOFTechnique = "_Basic_Instanced_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour = "ERMB_RENDER";
	string shadowGenTechnique = "_Basic_Instanced_ShadowGen";
	int    shadowGenDeferredID	= 0;
	string overdrawBehaviour = "ERMB_RENDER";
	string overdrawTechnique = "_Basic_Instanced_Overdraw";
	int    overdrawDeferredID	= 0;
	string meshVisBehaviour   = "ERMB_DONT_RENDER";
	bool   instancedShader = true;
	bool   disableAmbientShadow = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx BasicLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx BasicLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 BasicLowDetailVertexShader();
		PixelShader = compile ps_3_0 BasicLowDetailFragmentShader();
#endif
	}
}


technique _Basic_Instanced_ShadowGen
{
	pass Pass0
	{
			SHADOWMAP_STATES_ALPHATEST( 64 )
#if defined (_PS3_)
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx BasicInstancedTranslucentShadowGenVertexShader();
			PixelShader = compile sce_fp_rsx BasicInstancedTranslucentShadowGenFragmentShader();
#elif defined (_XBOX)			
			VertexShader = compile vs_3_0 BasicInstancedTranslucentShadowGenVertexShader();
			PixelShader = compile ps_3_0 BasicInstancedTranslucentShadowGenFragmentShader();
#else
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 BasicInstancedTranslucentShadowGenVertexShader();
			PixelShader = compile ps_3_0 BasicInstancedTranslucentShadowGenFragmentShader();
#endif
	}
}

technique _Basic_Instanced_ZPrime_DOF
{
	pass Pass0
	{
#if defined (_PS3_)
			AlphaTestEnable = true;
			ZEnable = true;
			ZWriteEnable = true;
		  AlphaFunc = int2(GEqual, 64);
			VertexShader = compile sce_vp_rsx BasicInstancedZPrimeDOFVertexShader();
			PixelShader = compile sce_fp_rsx BasicInstancedZPrimeDOFFragmentShader();
#else
			AlphaTestEnable = true;
			ZEnable = true;
			ZWriteEnable = true;
			AlphaRef = 64;
	    AlphaFunc = GreaterEqual;
			VertexShader = compile vs_3_0 BasicInstancedZPrimeDOFVertexShader();
			PixelShader = compile ps_3_0 BasicInstancedZPrimeDOFFragmentShader();
#endif
	}
}


technique _Basic_Instanced_Overdraw
{
	pass Pass0
	{
		ZEnable = false;
		ZWriteEnable = false;
		AlphaBlendEnable = true;

#if defined (_PS3_)
			BlendEquation=int(FuncAdd);
			BlendFunc=int2(One, One);
			VertexShader = compile sce_vp_rsx BasicVertexShader();
			PixelShader = compile sce_fp_rsx BasicInstancedOverdrawFragmentShader();
#else
			SrcBlend = ONE;
			DestBlend = ONE;
			BlendOp = ADD;
			VertexShader = compile vs_3_0 BasicVertexShader();
			PixelShader = compile ps_3_0 BasicInstancedOverdrawFragmentShader();
#endif
	}
}


technique _Basic_Instanced_Solid_ShadowGen
{
	pass Pass0
	{
			AlphaTestEnable = false;
	
#if defined (_PS3_)
			VertexShader = compile sce_vp_rsx BasicInstancedSolidShadowGenVertexShader();
			PixelShader = compile sce_fp_rsx BasicInstancedSolidShadowGenFragmentShader();
#elif defined(_XBOX) && !defined(SHADOWMAP_COLOUR)
			VertexShader = compile vs_3_0 BasicInstancedSolidShadowGenVertexShader();
			PixelShader = null;
#else
			VertexShader = compile vs_3_0 BasicInstancedSolidShadowGenVertexShader();
			PixelShader = compile ps_3_0 BasicInstancedSolidShadowGenFragmentShader();
#endif
	}
}

